// Louis-Clément LEFÈVRE
//
// Une page de notes à partir d'un fichier CSV

#import "hoche.typ": *

// besoin d'un titre global pour le PDF
// mais chaque liste de notes est sur sa propre page
#let hoche-notes(
  titre-PDF: none,
  body
) = {
  show: hoche.with(
    titre: none,
    titre-PDF: titre-PDF,
    no-header: true,
    extra-keywords: ("notes", )
  )
  set heading(numbering: (..nums) => numbering("I.1", ..(nums.pos().slice(1))))
  show heading.where(level: 1): it => {
    pagebreak(weak: true)
    v(-0.5cm)
    align(center,
      text(size: 18pt, weight: "bold",
        it.body
      )
    )
  }
  body
}

// Le fichier CSV doit contenir
// * une colonne "NOM" et une "Prénom"
// * une colonne de notes
// * éventuellement : une colonne de notes brutes, sert à faire le classement, sinon on prend la note /20
// * une ligne de moyennes, avec le prénom "moyenne" et le nom vide
// * une ligne de total, avec le prénom "total" et le nom vide

// dans un premier temps, on détecte tout cela
// les absents, le nombre total d'élèves, et le classement sont recalculés par le programme Typst

#let page-notes(
  titre,                // obligatoire : le titre à afficher
  fichier,              // obligatoire : le fichier CSV à charger
  style: "alpha",       // "alpha" ou bien "rang"
  style-couleur: "brut",      // "brut" ou "note" ou "rang" qui déterminent la couleur
  no-brut: false,       // pas de note brute
  tableau-largeur-nom: 16.5em, // largeur de la colonne NOM Prénom
  //tableau-inset: 3.6pt, // inset dans le tableau, réduire si ça ne rentre pas sur une seule page
  tableau-inset: 3.5pt, // inset dans le tableau, réduire si ça ne rentre pas sur une seule page
  // options qui sont détectées automatiquement du fichier CSV
  total: none,
  total-présent: none,
  note-max: none,
  note-max-brut: none,
  note-moyenne: none,
  note-moyenne-brut: none,
  // colonnes du fichier CSV
  CSV-col-rang: 0,
  CSV-col-nom: 1,
  CSV-col-prénom: 2,
  CSV-col-brut: 3,
  CSV-col-note: 4,
) = {
  // chargement du fichier CSV
  assert(fichier != none, message: "Il faut un fichier CSV des notes")
  assert(style == "alpha" or style == "rang")
  assert(style-couleur == "brut" or style-couleur == "note" or style-couleur == "rang" or style-couleur == "none")
  assert(not(style-couleur == "brut") or not(no-brut))

  let tableau = csv(fichier)

  // fonctions utiles
  let est_élève(x) = {
    x.at(CSV-col-nom) != "" and not(x.at(CSV-col-nom) == "NOM" and x.at(CSV-col-prénom) == "Prénom")
  }
  let est_absent(x) = {
    if not(no-brut) { x.at(CSV-col-brut) == "ae" }
    else { x.at(CSV-col-note) == "ae" }
  }
  // let protège_nom(s) = {
  //   s.replace("--", "\u{2010}\u{2010}")
  // }

  // compte du nombre total d'élèves, puis de présents
  let total = if total == none {
    tableau.fold(0, (n, x) => {
      if est_élève(x) { n + 1 }
      else { n }
    })
  } else { total }
  let total-présent = if total-présent == none {
    tableau.fold(0, (n, x) => {
      if est_élève(x) and not(est_absent(x)) { n + 1 }
      else { n }
    })
  } else { total-présent }
  assert(total-présent <= total)

  // tableau des notes, presque prêt à être inséré
  // refaire avec un dictionnaire
  let notes = if style == "alpha" {
    tableau.filter(est_élève)
  } else {
    tableau.filter(est_élève).sorted(key: x => {
      if est_absent(x) {
        total + 1
      } else {
        let r = int(x.at(CSV-col-rang))
        assert(r <= total-présent)
        r
      }
    })
  }.map(x => {(
    rang: x.at(CSV-col-rang),
    nom: x.at(CSV-col-nom),
    prénom: x.at(CSV-col-prénom),
    note: x.at(CSV-col-note),
    brut: if not(no-brut) { x.at(CSV-col-brut) } else { none }
    // protège_nom(x.at(CSV-col-nom)) + " " + protège_nom(x.at(CSV-col-prénom)),
  )})

  // récupérer les statistiques
  let ligne-max = tableau.find(x => {
    not(est_élève(x)) and (x.at(CSV-col-nom) == "" and x.at(CSV-col-prénom) == "total")
  })
  let note-max = if note-max == none {
    if ligne-max != none and ligne-max.at(CSV-col-note) != "" { ligne-max.at(CSV-col-note)}
    else { none }
  } else { note-max }
  let note-max-brut = if not(no-brut) {
    if note-max-brut == none {
      if ligne-max != none and ligne-max.at(CSV-col-brut) != "" { ligne-max.at(CSV-col-brut) }
      else { none }
    } else { note-max-brut }
  } else { none }
  let ligne-moyenne = tableau.find(x => {
    not(est_élève(x)) and (x.at(CSV-col-nom) == "" and x.at(CSV-col-prénom) == "moyenne")
  })
  let note-moyenne = if note-moyenne == none {
    if ligne-moyenne != none and ligne-moyenne.at(CSV-col-note) != "" { ligne-moyenne.at(CSV-col-note) }
    else { none }
  } else { note-moyenne }
  let note-moyenne-brut = if not(no-brut) {
    if note-moyenne-brut == none {
      if ligne-moyenne != none and ligne-moyenne.at(CSV-col-brut) != "" { ligne-moyenne.at(CSV-col-brut) }
      else { none }
    } else { note-moyenne-brut }
  } else { none }

  // essai pour la couleur : selon les notes et pas les rangs
  //  let note-min-couleur = calc.min(..notes.filter(x => x.note != "ae").map(x => float(if not(no-brut) { x.brut.replace(",", ".") } else { x.note.replace(",", ".") })))
  //  let note-min-couleur = 0
  let couleur-max = if style-couleur == "brut" {
    calc.max(..notes.filter(x => x.note != "ae").map(x => float(x.brut.replace(",", "."))))
  } else if style-couleur == "note" {
    calc.max(..notes.filter(x => x.note != "ae").map(x => float(x.note.replace(",", "."))))
  } else if style-couleur == "rang" {
    total-présent
  }
  // let note-max-couleur = calc.max(..notes.filter(x => x.note != "ae").map(x => float(if not(no-brut) { x.brut.replace(",", ".") } else { x.note.replace(",", ".") })))
  // let note-amplitude-couleur = note-max-couleur - note-min-couleur
  // ou les passer comme arguments
  // choisir comme min 0 ? ne change rien…

  // à partir d'ici toutes les données sont enregistrées
  // début du formatage, ne dépend plus du fichier CSV du tout
  // pagebreak(weak: true)
  heading(level: 1, titre)
  set text(hyphenate: false)
  align(center, table(
    columns: (3em, tableau-largeur-nom, ..(if not(no-brut) { (3.5em, ) } else { none }), 3.5em, ),
    // principal paramètre à ajuster pour tout tenir sur une seule page
    inset: tableau-inset,
    stroke: 0.4pt,
    align: horizon,
    fill: (j, i) => {
      if i == 0 or i == total+1 { gray }
      else { none }
    },
    // le haut
    table.header(repeat: true,
      table.cell(align: center, text(size: 8pt, weight: "bold",
        [Rang] + if total-présent != none [\ /~#total-présent]
      )),
      table.cell(align: center, text(weight: "bold",
        [NOM Prénom]
      )),
      ..(if not(no-brut) {(
        table.cell(align: center, text(size: 8pt, weight: "bold",
          [Brut] + if note-max-brut != none [\ /~#note-max-brut]
        )), )
      } else { none }),
      table.cell(align: center, text(size: 8pt, weight: "bold",
        [Note] + if note-max != none [\ /~#note-max]
      )),
    ),
    // la liste de notes
    ..(notes.map(x => {
      let couleur = if style == "alpha" { none } else {
        // améliorer : cas où il n'y a pas de note brute
        if x.note != "ae" {
          if style-couleur == "brut" {
            let n = float(x.brut.replace(",", "."))
            rgb(int((couleur-max - n) * 255 / couleur-max), int(n * 255 / couleur-max), 0).lighten(50%)
            // rgb(int((couleur-max - n) * 255 / couleur-max), int(n * 255 / couleur-max), 0).lighten(50%)
          } else if style-couleur == "note" {
            let n = float(x.note.replace(",", "."))
            rgb(int((couleur-max - n) * 255 / couleur-max), int(n * 255 / couleur-max), 0).lighten(50%)
          } else if style-couleur == "rang" {
            let n = int(x.rang)
            rgb(calc.quo((n - 1) * 255, couleur-max - 1), calc.quo((couleur-max - n) * 255, couleur-max - 1), 0).lighten(50%)
          }
        } else { silver }
      }
      (
      table.cell(align: right, fill: couleur,
          if x.rang != "ae" { x.rang } else { "abs"}
        ),
        table.cell(align: left, fill: couleur,
          x.nom.replace("--", "\u{2010}\u{2010}") + " " + x.prénom
        ),
        ..(if not(no-brut) {(
          table.cell(align: right, fill: couleur,
            if x.brut != "ae" { x.brut } else { "abs"}
          ),
        )} else { none }),
        table.cell(align: right, fill: couleur,
          if x.note != "ae" { x.note } else { "abs" }
        ),
      )
    })).flatten(),
    // le bas
    table.footer(repeat: false,
      table.cell(none, fill: none, stroke: none),
      table.cell(align: left, text(weight: "bold",
        [Moyenne]
      )),
      ..(if not(no-brut) {(
        table.cell(align: right,
          [#note-moyenne-brut]
        ),
      )} else { none }),
      table.cell(align: right,
        [#note-moyenne]
      ),
    )
  ))
}
