// Louis-Clément LEFÈVRE
//
// style pour un cahier de texte

#import "hoche.typ": *

// sauvegarde la fonction built-in
#let _type = type

#let hoche-cdt(
  année: "2024" + sym.dash.en + "2025",
  body
) = {
  show: hoche.with(
    titre: "Cahier de texte" + linebreak() + année,
    titre-header: "Cahier de texte",
    titre-PDF: "Cahier de texte" + " " + année,
    code: "CDT",
    année: année,
    extra-keywords: ("mathématiques", "cahier de texte", ),
  )
  set heading(numbering: none)
  body
}

// couleurs
// l'idée serait de pouvoir personnaliser le dictionnaire (show rule ?)
#let dictionnaire-couleurs = (
  "cours": red.lighten(80%),
  "TD": blue.lighten(80%),
  "TP": green.lighten(80%),
  "DS": silver.lighten(80%),
  "spécial": yellow.lighten(80%),
)

// pour les statistiques
#let total-cours = state("total-cours", 0)
#let total-TD = state("total-TD", 0)
#let total-TP = state("total-TP", 0)
#let total-DS = state("total-DS", 0)
#let total-heures-cours = state("total-heures-cours", duration(hours: 0))
#let total-heures-TD = state("total-heures-TD", duration(hours: 0))
#let total-heures-TP = state("total-heures-TP", duration(hours: 0))
#let total-heures-DS = state("total-heures-DS", duration(hours: 0))

// met à jour les compteurs
#let update-statistiques(type, durée) = {
  if durée != none {
    let (h, m) = durée
    let d = duration(hours: h, minutes: m)
    if type == "cours" {
      total-cours.update(x => x + 1)
	  total-heures-cours.update(x => x + d)
	} else if type == "TD" {
  	  total-TD.update(x => x + 1)
	  total-heures-TD.update(x => x + d)
	} else if type == "TP" {
	  total-TP.update(x => x + 1)
	  total-heures-TP.update(x => x + d)
	} else if type == "DS" {
	  total-DS.update(x => x + 1)
	  total-heures-DS.update(x => x + d)
    }
  }
}

#let get-couleur(type) = {
  if dictionnaire-couleurs != none {
    if type in dictionnaire-couleurs {
      dictionnaire-couleurs.at(type)
    } else {
      panic("type de séance non reconnu", type)
    }
  }
}

// "durée" est de type duration ou bien tuple (heure, minute)
// mode 1 : dans la colonne gauche
// mode 2 : statistiques
#let print-durée(mode: 1, x2: false, durée) = {
  let (h, m) = {
    if _type(durée) == array and durée.len() == 2 { durée }
    else if _type(durée) == duration {
      let H = durée.hours()
      let R = durée - duration(hours: calc.floor(H))
      (calc.floor(H), calc.floor(R.minutes()))
    }
  }
  if h > 0 {
    str(h) + sym.space.nobreak
    if h > 1 { "heures" }
    else { "heure" }
  }
  if m > 0 {
    if h > 0 { sym.space }
    str(m)
    if mode == 2 or h == 0 {
      sym.space.nobreak
      if m > 1 { "minutes" }
      else { "minute" }
    }
  }
  if x2 {
    if mode == 1 and (h != 0 or m != 0) { sym.space }
    else if mode == 2 and (h != 0 or m != 0) { ", " }
    if mode == 1 or (h != 0 or m != 0) { $times 2$ }
  }
}

// crée une séance, renvoie une liste de cellules prêtes à être insérées
#let séance(
  type: none,
  date: none,
  durée: none,
  x2: false,
  spécial: false,
  ..L,
) = {
  // les éléments de L viennent par paire
  assert(calc.even(L.pos().len()), message: "Il faut un nombre pair d'éléments séance.")

  // gestion de la durée : c'est toujours un tuple (heures, minutes)
  let durée = if durée != none {
    if _type(durée) == int { (durée, 0) }
    else if _type(durée) == array and durée.len() == 1 { (durée.at(0), 0) }
    else if _type(durée) == array and durée.len() == 2 { durée }
    else if _type(durée) == str { durée }
    else { panic("durée non reconnue", durée) }
  }

  // cellules de droite
  let D = L.pos().chunks(2, exact: true).map( ((titre, contenu)) => {(
    table.cell(
      breakable: false,
      fill: get-couleur(type),
      {
        set text(hyphenate: false)
        set par(justify: false)
        titre
      }
    ),
    if contenu != [] {
      table.cell(
        breakable: false,
        fill: get-couleur(type),
        stroke: (top: (dash: "dotted")),
        {
          set text(size: 8pt)
          set list(tight: true, marker: none, body-indent: 0pt)
          contenu
        }
      )
    } else {
      none
    }
  )}).flatten().filter( x => x != none)
  let n = D.len()
  
  // cellule de gauche
  let G = table.cell(
    breakable: false,
    fill: get-couleur(type),
    rowspan: n,
    {
      set text(hyphenate: false)
      set par(justify: false)
	  if durée != none and _type(durée) != str { update-statistiques(type, durée) }
      if spécial { "*" + if date != none { " " } }
      date
      // text(weight: "bold", date)
      if (durée != none or _type(durée) == str) or x2 {
        linebreak()
        set text(size: 8pt)
        if _type(durée) != str { print-durée(mode: 1, x2: x2, durée) }
        else { durée }
      }
    }
  )
  return (G, ) + D
}

// prend en argument une liste de cellules déjà prêtes
#let semaine(numéro: none, date: none, ..L) = {
  set table.cell(breakable: false)
  [= Semaine~#numéro~: #date]
  table(
    columns: (9em, 1fr),
    stroke: 0.4pt,
    ..(L.pos().flatten())
  )
}

#let statistiques() = {
  let print-séances(n) = {
    if n > 0 {
      str(n) + sym.space.nobreak
      if n > 1 { "séances" }
      else { "séance" }
      ","
    }
  }
  [= Statistiques totales]
  set table.cell(breakable: false)
  context {
    table(
      columns: (9em, 1fr),
      stroke: 0.4pt,
      fill: (_, i) => (get-couleur("cours"), get-couleur("TD"), get-couleur("TP"), get-couleur("DS")).at(i),
      [*Cours*], [#print-séances(total-cours.final()) #print-durée(mode: 2, x2: false, total-heures-cours.final())],
	  [*TD*], [#print-séances(total-TD.final()) #print-durée(mode: 2, x2: true, total-heures-TD.final())],
	  [*TP*], [#print-séances(total-TP.final()) #print-durée(mode: 2, x2: true, total-heures-TP.final())],
	  [*DS*], [#print-séances(total-DS.final()) #print-durée(mode: 2, x2: false, total-heures-DS.final())],
    )
  }
}
